﻿//----------------------------------------------------------
// Copyright (C) Microsoft Corporation. All rights reserved.
//----------------------------------------------------------
// TFS.Core.Ajax.js
//
/*globals endWait, showWait, startWait */
/// <reference path="~/Scripts/DevTime.js" />
/// <reference path="~/Scripts/jquery-1.6.2.js" />
/// <reference path="~/Scripts/MicrosoftAjax-4.0.0.0.debug.js"/>
/// <reference path="~/Scripts/TFS/TFS.debug.js" />

TFS.module("TFS.Core.Ajax", ["TFS.Core", "TFS.Diag", "TFS.Core.Utils", "TFS.Resources.Common"], function () {
    var Module,
        unloading = false,
        unloadRequested = false,
        requestId = 1, /* unique request id */
        defaultTimeout = 300000, /*5 mins*/
        handleError = TFS.handleError,
        moduleExceptions = {
            AjaxException: "TFS.Core.Ajax.AjaxException",
            AjaxTimeoutException: "TFS.Core.Ajax.AjaxTimeoutException",
            NetworkException: "TFS.Core.Ajax.NetworkException"
        },
        CommonResources = TFS.Resources.Common,
        getAntiForgeryTokenValue = TFS.Core.getAntiForgeryTokenValue,
        getAntiForgeryTokenValue2 = TFS.Core.getAntiForgeryTokenValue2,
        requests = {},
        Utils = TFS.Core.Utils,
        Core = TFS.Core;

    Error.ajax = function (xhr, status) {
        var err = Error.create(String.format(CommonResources.AjaxRequestFailedWithStatus, status), { name: moduleExceptions.AjaxException, response: xhr.responseText, xhr: xhr });
        err.popStackFrame();
        return err;
    };

    Error.ajaxTimeout = function (xhr) {
        var err = Error.create(CommonResources.AjaxRequestTimedOut, { name: moduleExceptions.AjaxTimeoutException, status: xhr.status, xhr: xhr });
        err.popStackFrame();
        return err;
    };

    Error.network = function (xhr) {
        var err = Error.create(CommonResources.NetworkConnectionUnavailable, { name: moduleExceptions.NetworkException, status: xhr.status, xhr: xhr});
        err.popStackFrame();
        return err;
    };

    function endRequest(data, textStatus, xhr, callback, requestContext) {

        /// <summary>Determine if the response came from a location that was not the actual content we requested.
        ///  Note: X-Tfs-Location will have the actual location the content was returned from.
        /// </summary>

        var targetLocation,
            replyLocationStart,
            replyLocationEnd,
            newLocation;

        TFS.Diag.logTracePoint("Ajax.response-received", requestContext.requestId);

        // Mark the request as being complete.
        requestContext.isComplete = true;

        delete requests[requestContext.requestId];

        // End any active wait session associated with this request
        if (requestContext.options != null && requestContext.options.wait != null) {
            endWait(requestContext);
        }

        // Determine if the response came from a location that was
        // not the actual content we requested.
        // Note: X-Tfs-Location will have the actual location the
        // content was returned from.
        if (textStatus !== 'timeout' && xhr && xhr.status === 203) {

            targetLocation = xhr.getResponseHeader('X-Tfs-Location');
            if (targetLocation) {

                // If the document contains a replyTo we need to replace
                // it with our location
                replyLocationStart = targetLocation.indexOf('reply_to=');
                if (replyLocationStart !== -1) {
                    replyLocationEnd = targetLocation.indexOf('&', replyLocationStart);

                    // Replace the current reply with the current location
                    newLocation = targetLocation.substring(0, replyLocationStart);
                    if (replyLocationEnd !== -1) {
                        newLocation += targetLocation.subString(replyLocationEnd);
                    }

                    // Trim an trailing &'s from the current URL.
                    while (newLocation.charAt(newLocation.length - 1) === '&') {
                        newLocation = newLocation.substring(0, newLocation.length - 1);
                    }

                    // Add the current reply_to location
                    targetLocation = newLocation + '&reply_to=' + encodeURIComponent(window.location);
                }

                // Navigate the page to the targetLocation
                window.location = targetLocation;
                return;
            }
        }

        if (callback) {
            try {
                callback(data, textStatus, xhr);
            }
            catch (ex) {
                TFS.Diag.logTracePoint("Ajax.callback-exception", ex);
                handleError(ex);
            }
        }

        TFS.Diag.logTracePoint("Ajax.callback-complete", requestContext.requestId);
        if (requestContext.progressActionId) {
            TFS.globalProgressIndicator.actionCompleted(requestContext.progressActionId);
        }

        // Log a specific trace point one was specified
        if (requestContext.options && requestContext.options.tracePoint) {
            TFS.Diag.logTracePoint(requestContext.options.tracePoint);
        }
    }

    function wrapSuccessCallback(requestContext, callback) {
        return function (data, textStatus, xhr) {
            endRequest(data, textStatus, xhr, callback, requestContext);
        };
    }

    function wrapErrorCallback(requestContext, callback) {
        return function (xhr, textStatus, errorThrown) {

            endRequest(null, textStatus, xhr, function () {
                //not a non authoritative response then handle it regular way
                var serverError, timerId;

                function handleNetworkError(){
                    if (!unloadRequested && !unloading) {
                        handleError(Error.network(xhr), callback, Module);
                    }
                }

                if (textStatus === "timeout") {
                    handleError(Error.ajaxTimeout(xhr), callback, Module);
                }
                else if(xhr.status == 0){
                    if (!unloadRequested && !unloading) {
                        Core.delay(this, 2000, function () {
                            //defer error event for 2 seconds
                            //This is a temporary remedy caused by a Chrome bug
                            //When browsers back and forward buttons are used Chrome cancels our requests with status == 0
                            //before firing unbeforeunload event. This causes an annoyance in our pages due to network error message.
                            //http://code.google.com/p/chromium/issues/detail?id=4422
                            if (!unloadRequested && !unloading) {
                                handleError(Error.network(xhr), callback, Module);
                            }
                        });
                    }
                }
                else if (xhr.status >= 12000 && xhr.status < 13000) {
                    if (!unloadRequested && !unloading) {
                        handleError(Error.network(xhr), callback, Module);
                    }
                } else {
                    if (unloading && textStatus === "abort") {
                        return;
                    }

                    try {
                        serverError = Utils.parseMSJSON(xhr.responseText, false);
                    }
                    catch (exc) {
                    }

                    if (serverError) {
                        handleError(Error.serverError(serverError), callback, Module);
                    }
                    else {
                        handleError(Error.ajax(xhr, xhr.statusText || xhr.status), callback, Module);
                    }
                }
            }, requestContext);
        };
    }

    // Perform any pre-request setup before the request is made
    function beginRequest(url, data, options) {

        var requestContext, actionName;

        // Create an object that contains context for this request
        requestContext = {

            // Mark the request as not being complete
            isComplete: false,

            // Assign the request a request Id
            requestId: requestId++,

            // If the client asked for a wait animation we will start it now.
            options: options
        };

        if (options && options.wait) {
            startWait(requestContext);
        }

        actionName = url + " (" + requestContext.requestId + ")";
        TFS.Diag.logTracePoint("Ajax.request-started", actionName);

        if (!options || options.showGlobalProgressIndicator !== false) {
            requestContext.progressActionId = TFS.globalProgressIndicator.actionStarted(actionName);
        }

        requests[requestContext.requestId] = requestContext;

        return requestContext;
    }

    function resizeWait(requestContext) {

        var wait = requestContext.options.wait;

        // Update the size and position of the wait element to cover the target properly.
        if (!wait.entireWindow) {

            wait.element.css('margin-top', -parseInt(wait.target.css('padding-top'), 10));
            wait.element.css('margin-left', -parseInt(wait.target.css('padding-left'), 10));
            wait.element.height(wait.target.outerHeight());
            wait.element.width(wait.target.outerWidth());
        }

        // Update the position of the message element if one is shown
        if (wait.msgElement) {
            wait.msgElement.css('top', wait.element.position().top +
                (wait.element.height() - wait.msgElement.height()) / 2 +
                parseInt(wait.element.css('margin-top'), 10));
            wait.msgElement.css('left', wait.element.position().left +
                (wait.element.width() - wait.msgElement.width()) / 2 +
                parseInt(wait.element.css('margin-left'), 10));
        }
    }

    function startWait(requestContext) {
        var wait = requestContext.options.wait;

        // Hook up the function used to end the wait
        wait.end = function () {
            if (wait.element) {

                // If it is complete when the timer goes off kill the waitElement
                if (requestContext.isComplete && !wait.timeout) {

                    if (wait.fade !== false) {
                        wait.element.css('cursor', 'auto');  // see vstspioneer bug 884483: Cursor is not reset in browsers when element is removed until a mousemove
                        wait.element.fadeOut('fast', function () {
                            wait.element.remove();
                            wait.element = null;
                        });

                        if (wait.msgElement) {
                            wait.msgElement.css('cursor', 'auto');
                            wait.msgElement.fadeOut('fast', function () {
                                wait.msgElement.remove();
                                wait.msgElement = null;
                            });
                        }
                    }
                    else {
                        wait.element.css('cursor', 'auto');
                        wait.element.remove();
                        wait.element = null;

                        if (wait.msgElement) {
                            wait.msgElement.css('cursor', 'auto');
                            wait.msgElement.remove();
                            wait.msgElement = null;
                        }
                    }

                    // Remove the resize binder for moving the wait element
                    $(window).unbind('resize.' + requestContext.requestId);
                }
            }
        };

        // Make sure we have a showDelay set
        if (!wait.showDelay) {
            wait.showDelay = 250;
        }

        // If no target was specified use the entire window as the wait target
        if (!wait.target) {
            wait.entireWindow = true;
            wait.target = $('body');
            extraStyles = "height:100%; width:100%;";
        }

        // Hide any existing waits on children
        wait.target.children('.wait-element').hide();
        wait.target.children('.wait-box').hide();

        // If a showDelay is specified, call showWait after showDelay time
        if (wait.showDelay !== 0) {
            requestContext.showTimer = Core.delay(this, wait.showDelay, function () {
                delete requestContext.showTimer;
                showWait(requestContext);
            });
        }
        else {
            showWait(requestContext);
        }
    }

    function endWait(requestContext) {
        requestContext.options.wait.end();

        // Clear delayed show timer
        if (requestContext.showTimer) {
            requestContext.showTimer.cancel();
            delete requestContext.showTimer;
        }
    }

    function showWait(requestContext) {
        var msgContent = '',
            extraStyles = '',
            wait = requestContext.options.wait;

        // Determine what content is required for wait message
        if (wait.image) {
            msgContent += '<img class="wait-image" src="' + wait.image + '" />';
        }

        if (wait.message) {
            msgContent += '<div class="wait-message">' + wait.message + '</div>';
        }

        // If the caller wants animation and or a message we need to configure it
        if (msgContent.length > 0) {

            wait.target.prepend('<div class="wait-box">' + msgContent + '</div>');
            wait.msgElement = wait.target.children('.wait-box').first();
        }

        // Set the custom background color if one was specified
        if (wait.backgroundcolor) {
            extraStyles += 'background-color: ' + wait.backgroundcolor + ';';
        }

        // Build the waitElement and make sure it is sized properly
        wait.target.prepend('<div class="wait-element" style="' + extraStyles + '"></div>');
        wait.element = wait.target.children('.wait-element').first();

        // Make sure the wait element is laid out properly
        resizeWait(requestContext);

        // Attach to the window resize event to update our position
        $(window).bind('resize.' + requestContext.requestId, function () {
            resizeWait(requestContext);
        });

        // Make sure we have at least a minimal lifetime if it wasn't explicitly set
        if (!wait.minLifetime) {
            wait.minLifetime = 100;
        }

        // Don't allow the session to expire until the timeout.
        // This prevents flashing.
        wait.timeout = Core.delay(this, wait.minLifetime, function () {
            delete wait.timeout;
            wait.end();
        });
    }


    function msJSONFilter(data, type) {
        var parsedData;
        //Bug 814852, Security: unicode character  'LINE SEPARATOR' (U+2028) can break queries, work items, alerts page, and maybe others
        //Eval based JSON parser is broken when unicode line separator is observed in strings.
        if(typeof data === "string"){
            data = data.replace(/\u2028/g, "\\u2028");
        }

        parsedData = Utils.parseMSJSON(data, false);

        TFS.Diag.assert(!$.isArray(parsedData), "Received JSON data from the server which is an array.  Sending JSON arrays from the server leaves us open to JSON Hijacking attacks.  Ensure that the server code is making use of the SecureJsonResult class.");

        // If the data object has a wrapped array property, then unwrap the array.
        // The SecureJsonResult on the server will wrap any arrays that are being sent
        // back to the client in an object to prevent JSON Hijacking attacks.  To make
        // it this transparent to the code consuming the results on the client, we
        // unwrap the array here.
        if (parsedData.hasOwnProperty("__wrappedArray")) {
            parsedData = parsedData.__wrappedArray;
        }

        return parsedData;
    }

    function beforeUnload(){
        unloadRequested = true;
        Core.delay(this, 2000, function () {
            // reset the unload requested flag, as the user could cancel the unload
            // (e.g. in the case of a dirty document which causes a Stay on this page? prompt.)
            unloadRequested = false;
        });
    }

    $(window).bind("beforeunload", beforeUnload);

    $(window).unload(function () {
        unloading = true;

        $.each(requests, function (reqId, requestContext) {
            if (requestContext.xhr) {
                try {
                    //abort in-progress requests to prevent memory leak in IE.
                    requestContext.xhr.abort();
                }
                catch (ex) {
                }
            }
        });
    });

    Module = $.extend({
        defaultTimeout: defaultTimeout,
        timeout: defaultTimeout,

        getMSJSON: function (url, data, callback, errorCallback, ajaxOptions) {

            // Perform any pre work needed for the reuqest.
            var requestContext = beginRequest(url, data, ajaxOptions);

            requestContext.xhr = jQuery.ajax($.extend({
                type: "GET",
                url: url,
                data: data,
                success: wrapSuccessCallback(requestContext, callback),
                error: wrapErrorCallback(requestContext, errorCallback),
                dataType: "json",
                //                    dataFilter: msJSONFilter,
                converters: { "text json": msJSONFilter },  // We convert in dataFilter
                traditional: true,
                timeout: Module.timeout
            }, ajaxOptions));

            return requestContext;
        },

        postMSJSON: function (url, data, callback, errorCallback, ajaxOptions) {
            var token2;

            data = $.extend(data || {}, { __RequestVerificationToken: getAntiForgeryTokenValue() });
            token2 = getAntiForgeryTokenValue2();
            if (token2) {
                data = $.extend(data, {__RequestVerificationToken2: token2});
            }
            // Perform any pre work needed for the reuqest.
            var requestContext = beginRequest(url, data, ajaxOptions);

            try {
                requestContext.xhr = jQuery.ajax($.extend({
                    type: "POST",
                    url: url,
                    data: data,
                    success: wrapSuccessCallback(requestContext, callback),
                    error: wrapErrorCallback(requestContext, errorCallback),
                    dataType: "json",
//                    dataFilter: msJSONFilter,
                    converters: { "text json": msJSONFilter },  // We convert in dataFilter
                    traditional: true,
                    timeout: Module.timeout
                }, ajaxOptions));
            }
            catch (exception) {
                // Exceptions can be thrown by jQuery.ajax for things like invalid characters being sent in the data stream.
                errorCallback(exception);
            }

            return requestContext;
        },

        postHTML: function (url, data, callback, errorCallback, ajaxOptions) {
            var token2;

            data = $.extend(data, { __RequestVerificationToken: getAntiForgeryTokenValue() });
            token2 = getAntiForgeryTokenValue2();
            if (token2) {
                data = $.extend(data, { __RequestVerificationToken2: token2 });
            }

            // Perform any pre work needed for the reuqest.
            var requestContext = beginRequest(url, data, ajaxOptions);

            requestContext.xhr = jQuery.ajax($.extend({
                type: "POST",
                url: url,
                data: data,
                ajaxHandler: this,
                success: wrapSuccessCallback(requestContext, callback),
                error: wrapErrorCallback(requestContext, errorCallback),
                dataType: "html",
                traditional: true,
                cache: false
            }, ajaxOptions));

            return requestContext;
        },

        getHTML: function (url, data, callback, errorCallback, ajaxOptions) {

            // Perform any pre work needed for the reuqest.
            var requestContext = beginRequest(url, data, ajaxOptions);

            requestContext.xhr = jQuery.ajax($.extend({
                type: "GET",
                url: url,
                data: data,
                success: wrapSuccessCallback(requestContext, callback),
                error: wrapErrorCallback(requestContext, errorCallback),
                dataType: "html",
                traditional: true,
                cache: false
            }, ajaxOptions));

            return requestContext;
        },

        getJSONp: function (ajaxOptions) {

            // Perform any pre work needed for the reuqest.
            var requestContext = beginRequest(
                ajaxOptions && ajaxOptions.url,
                ajaxOptions && ajaxOptions.data,
                ajaxOptions);

            requestContext.xhr = jQuery.ajax($.extend({
                type: "GET",
                dataType: "jsonp",
                traditional: true,
                success: wrapSuccessCallback(requestContext, ajaxOptions && ajaxOptions.callback)
            }, ajaxOptions));

            return requestContext;
        },

        endRequest: endRequest,
        startWait: startWait,
        endWait: endWait
    }, moduleExceptions);
    return Module;
});

// SIG // Begin signature block
// SIG // MIIaowYJKoZIhvcNAQcCoIIalDCCGpACAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFC7y5cCwm7cy
// SIG // qHtGUOZqX5GpegcCoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSWMIIEkgIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIG4MBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBRQXou4xsbbzzLlbMvsZLpr
// SIG // 1o8jsTBYBgorBgEEAYI3AgEMMUowSKAugCwAVABGAFMA
// SIG // LgBDAG8AcgBlAC4AQQBqAGEAeAAuAGQAZQBiAHUAZwAu
// SIG // AGoAc6EWgBRodHRwOi8vbWljcm9zb2Z0LmNvbTANBgkq
// SIG // hkiG9w0BAQEFAASCAQB3DEXQc9CsmFe43gWKWwWXjiHN
// SIG // 93UPUZLJw2pg2XD3vxfEAkeqW2gxutetaNmb2BKgzo79
// SIG // 71BZjz1f9LnpbxM8aVSvT17wCPVLGLEXZKCcGO2fghjk
// SIG // 2Sd+mBBQmiPKoQcnmK62dh9+yAk9oJAkB6P/wKcU4JqZ
// SIG // JusmsMtjk+naCru7SXK5LuIni8iixEqXj6QVGUR5s5Ie
// SIG // p38hpYeW4LIaMWqQoUNu7YzT1Z2d9T3dnzgz1yeE8w+4
// SIG // fb0LdF9H7KmbDZTBJO0+O3rnO68Top1ZdAHBNd09vN+F
// SIG // YDfsVt7Nw6nz1G4zzpj7zKKxf6fgU9BIoN0SidI4tfM1
// SIG // FyoRm0k5oYICHzCCAhsGCSqGSIb3DQEJBjGCAgwwggII
// SIG // AgEBMIGFMHcxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
// SIG // YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
// SIG // VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xITAfBgNV
// SIG // BAMTGE1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQQIKYQKS
// SIG // SgAAAAAAIDAJBgUrDgMCGgUAoF0wGAYJKoZIhvcNAQkD
// SIG // MQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUxDxcNMTMw
// SIG // MzE1MDYzMzU0WjAjBgkqhkiG9w0BCQQxFgQUOOwpfds6
// SIG // gFu8Sp79QeCfZbPqhHQwDQYJKoZIhvcNAQEFBQAEggEA
// SIG // ERfs2LU7F8vH0wsuDTikbkI8t6DQCEuj1dKEaxp1rmLW
// SIG // v7IE+coW+PXoi99ENlf1XI9fQh7NIoyI/9Ln4GJnn/DA
// SIG // Ibzx31LbwWn0rJF1ckPAymNunA5PMYK+4WKVw75KLeXh
// SIG // eFUzEIYFn+kJsjui530uCw4mgZcGcTs63WxsMoKT+qVg
// SIG // GJmJ2QdSltEJBUHt8tSx7ERMra5xVp415j57GuGJtqrm
// SIG // wLY8jxD71qyDm39DrcrDtPZUhrvdfd6mT8YUBjcGOZJo
// SIG // nNSI19c68m0jmS07pkenLMEVgff+jw5aIXkkmTkX5ofI
// SIG // cSJxbU9Z3fYWNMKTmoqQdH3neWqfMI+OHg==
// SIG // End signature block
